"""
BHR Battery Diagnostics - Host Client Example

This script demonstrates how to communicate with the BHR Battery Tool
from a host computer via serial port.
"""

import serial
import struct
import time
from typing import Optional, Tuple

# Command definitions
CMD_CUSTOM = 0xFF
CMD_SCAN_I2C = 0x10
CMD_READ_VOLTAGE = 0x11
CMD_READ_CURRENT = 0x12
CMD_READ_TEMPERATURE = 0x13
CMD_READ_SOC = 0x14
CMD_READ_CAPACITY = 0x15
CMD_READ_CYCLE_COUNT = 0x16
CMD_READ_STATUS = 0x17
CMD_READ_MANUFACTURER = 0x18
CMD_READ_DEVICE_NAME = 0x19
CMD_READ_CHEMISTRY = 0x1A
CMD_READ_SERIAL = 0x1B
CMD_READ_CELL_VOLTAGES = 0x1C

class BatteryDiagnosticsTool:
    """Client for communicating with BHR Battery Diagnostics Tool"""
    
    def __init__(self, port: str, baudrate: int = 115200, timeout: float = 2.0):
        """
        Initialize the diagnostic tool client
        
        Args:
            port: Serial port name (e.g., 'COM3' on Windows, '/dev/ttyUSB0' on Linux)
            baudrate: Communication speed (default: 115200)
            timeout: Read timeout in seconds (default: 2.0)
        """
        self.ser = serial.Serial(port, baudrate, timeout=timeout)
        time.sleep(2)  # Wait for Arduino to reset
        
        # Read any startup messages
        while self.ser.in_waiting:
            print(self.ser.readline().decode('utf-8', errors='ignore').strip())
    
    def send_command(self, command: int, payload: bytes = b'') -> Optional[bytes]:
        """
        Send a custom command to the device
        
        Args:
            command: Command byte
            payload: Optional payload data
            
        Returns:
            Response data or None if error
        """
        # Build command packet: [CMD_CUSTOM, subcommand, payload_length, payload...]
        packet = struct.pack('BBB', CMD_CUSTOM, command, len(payload)) + payload
        
        self.ser.write(packet)
        self.ser.flush()
        
        # Wait for response (simple implementation)
        time.sleep(0.1)
        
        if self.ser.in_waiting:
            response = self.ser.read(self.ser.in_waiting)
            return response
        
        return None
    
    def scan_i2c(self) -> Tuple[bool, Optional[int]]:
        """
        Scan I2C bus for battery device
        
        Returns:
            Tuple of (found, address) where found is bool and address is int or None
        """
        print("Scanning for battery...")
        
        # Send SCAN command
        self.ser.write(b"SCAN\n")
        self.ser.flush()
        
        # Wait for response
        time.sleep(0.5)
        
        # Read response lines
        found = False
        address = None
        
        while self.ser.in_waiting:
            line = self.ser.readline().decode('utf-8', errors='ignore').strip()
            print(line)
            
            if "STATUS: OK" in line:
                found = True
            elif "ADDRESS: 0x" in line:
                # Extract address from line
                addr_str = line.split("0x")[1].strip()
                address = int(addr_str, 16)
        
        if found and address:
            return True, address
        
        return False, None
    
    def unseal_battery(self, key1: int = 0xFFFF, key2: int = 0xFFFF, register: int = 0x41) -> bool:
        """
        Unseal the battery with provided keys
        
        Args:
            key1: First unseal key (default: 0xFFFF)
            key2: Second unseal key (default: 0xFFFF)
            register: Battery register to use (default: 0x41)
        
        Returns:
            True if unsealed successfully, False otherwise
        """
        print(f"Attempting to unseal with Key1=0x{key1:04X}, Key2=0x{key2:04X}, Register=0x{register:02X}...")
        
        # Send UNSEAL command with parameters
        command = f"UNSEAL 0x{key1:04X} 0x{key2:04X} 0x{register:02X}\n"
        self.ser.write(command.encode())
        self.ser.flush()
        
        # Wait for response
        time.sleep(1.0)
        
        # Read response lines
        unsealed = False
        
        while self.ser.in_waiting:
            line = self.ser.readline().decode('utf-8', errors='ignore').strip()
            print(line)
            
            if "SUCCESS: Battery unsealed!" in line:
                unsealed = True
        
        return unsealed
    
    def read_voltage(self) -> Optional[float]:
        """Read battery voltage in volts"""
        response = self.send_command(CMD_READ_VOLTAGE)
        
        if response and len(response) >= 3 and response[0] == 1:
            voltage_mv = (response[1] << 8) | response[2]
            voltage_v = voltage_mv / 1000.0
            return voltage_v
        
        return None
    
    def read_current(self) -> Optional[float]:
        """Read battery current in amps (positive = charging, negative = discharging)"""
        response = self.send_command(CMD_READ_CURRENT)
        
        if response and len(response) >= 3 and response[0] == 1:
            current_raw = (response[1] << 8) | response[2]
            # Convert to signed int16
            if current_raw > 32767:
                current_raw -= 65536
            current_ma = current_raw
            current_a = current_ma / 1000.0
            return current_a
        
        return None
    
    def read_temperature(self) -> Optional[float]:
        """Read battery temperature in Celsius"""
        response = self.send_command(CMD_READ_TEMPERATURE)
        
        if response and len(response) >= 3 and response[0] == 1:
            temp_raw = (response[1] << 8) | response[2]
            # Temperature is in 0.1K units
            temp_c = (temp_raw / 10.0) - 273.15
            return temp_c
        
        return None
    
    def read_state_of_charge(self) -> Optional[int]:
        """Read battery state of charge as percentage (0-100)"""
        response = self.send_command(CMD_READ_SOC)
        
        if response and len(response) >= 3 and response[0] == 1:
            soc = (response[1] << 8) | response[2]
            return soc
        
        return None
    
    def read_capacity(self) -> Optional[Tuple[int, int]]:
        """
        Read battery capacity
        
        Returns:
            Tuple of (remaining_mah, full_charge_mah) or None
        """
        response = self.send_command(CMD_READ_CAPACITY)
        
        if response and len(response) >= 5 and response[0] == 1:
            remaining = (response[1] << 8) | response[2]
            full = (response[3] << 8) | response[4]
            return remaining, full
        
        return None
    
    def read_cycle_count(self) -> Optional[int]:
        """Read battery cycle count"""
        response = self.send_command(CMD_READ_CYCLE_COUNT)
        
        if response and len(response) >= 3 and response[0] == 1:
            cycles = (response[1] << 8) | response[2]
            return cycles
        
        return None
    
    def read_manufacturer(self) -> Optional[str]:
        """Read manufacturer name"""
        response = self.send_command(CMD_READ_MANUFACTURER)
        
        if response and len(response) >= 2 and response[0] == 1:
            name = response[1:].decode('utf-8', errors='ignore').strip('\x00')
            return name
        
        return None
    
    def read_device_name(self) -> Optional[str]:
        """Read device name"""
        response = self.send_command(CMD_READ_DEVICE_NAME)
        
        if response and len(response) >= 2 and response[0] == 1:
            name = response[1:].decode('utf-8', errors='ignore').strip('\x00')
            return name
        
        return None
    
    def read_chemistry(self) -> Optional[str]:
        """Read battery chemistry"""
        response = self.send_command(CMD_READ_CHEMISTRY)
        
        if response and len(response) >= 2 and response[0] == 1:
            chemistry = response[1:].decode('utf-8', errors='ignore').strip('\x00')
            return chemistry
        
        return None
    
    def read_cell_voltages(self) -> Optional[list]:
        """Read individual cell voltages in volts"""
        response = self.send_command(CMD_READ_CELL_VOLTAGES)
        
        if response and len(response) >= 9 and response[0] == 1:
            cells = []
            for i in range(4):
                cell_mv = (response[1 + i*2] << 8) | response[2 + i*2]
                if cell_mv > 0:  # Only include non-zero cells
                    cells.append(cell_mv / 1000.0)
            return cells
        
        return None
    
    def print_all_info(self):
        """Print all available battery information"""
        print("\n" + "="*50)
        print("Battery Information")
        print("="*50)
        
        manufacturer = self.read_manufacturer()
        if manufacturer:
            print(f"Manufacturer:     {manufacturer}")
        
        device_name = self.read_device_name()
        if device_name:
            print(f"Device:           {device_name}")
        
        chemistry = self.read_chemistry()
        if chemistry:
            print(f"Chemistry:        {chemistry}")
        
        print("\n" + "-"*50)
        print("Current Status")
        print("-"*50)
        
        voltage = self.read_voltage()
        if voltage is not None:
            print(f"Voltage:          {voltage:.3f} V")
        
        current = self.read_current()
        if current is not None:
            print(f"Current:          {current:.3f} A")
        
        temp = self.read_temperature()
        if temp is not None:
            print(f"Temperature:      {temp:.1f} °C")
        
        soc = self.read_state_of_charge()
        if soc is not None:
            print(f"State of Charge:  {soc} %")
        
        capacity = self.read_capacity()
        if capacity:
            remaining, full = capacity
            print(f"Capacity:         {remaining} / {full} mAh")
        
        cycles = self.read_cycle_count()
        if cycles is not None:
            print(f"Cycle Count:      {cycles}")
        
        cells = self.read_cell_voltages()
        if cells:
            print("\n" + "-"*50)
            print("Cell Voltages")
            print("-"*50)
            for i, voltage in enumerate(cells, 1):
                print(f"Cell {i}:           {voltage:.3f} V")
        
        print("="*50 + "\n")
    
    def close(self):
        """Close the serial connection"""
        self.ser.close()


def main():
    """Example usage"""
    import sys
    
    # Configure your serial port here
    PORT = 'COM3'  # Change this to your port (e.g., 'COM3', '/dev/ttyUSB0')
    
    if len(sys.argv) > 1:
        PORT = sys.argv[1]
    
    try:
        print(f"Connecting to {PORT}...")
        tool = BatteryDiagnosticsTool(PORT)
        
        # Scan for battery
        found, address = tool.scan_i2c()
        
        if found:
            # Unseal battery with custom keys
            # Default: key1=0xFFFF, key2=0xFFFF, register=0x41
            # You can customize: tool.unseal_battery(key1=0x0414, key2=0x3672, register=0x00)
            unsealed = tool.unseal_battery(key1=0xFFFF, key2=0xFFFF, register=0x41)
            
            if unsealed:
                print("\nBattery unsealed successfully!\n")
            
            # Read and display all information
            tool.print_all_info()
            
            # You can also read individual parameters:
            # voltage = tool.read_voltage()
            # print(f"Voltage: {voltage} V")
        else:
            print("No battery detected. Please check connections.")
        
        tool.close()
        
    except serial.SerialException as e:
        print(f"Error: {e}")
        print(f"Could not open serial port {PORT}")
        print("Please check:")
        print("  1. The correct port name")
        print("  2. The device is connected")
        print("  3. No other program is using the port")
    except Exception as e:
        print(f"Unexpected error: {e}")
        import traceback
        traceback.print_exc()


if __name__ == "__main__":
    main()
