#!/bin/bash
# BHR Battery Diagnostics Tool - Firmware Flasher for Linux/Mac
# Requires: Python 3 and esptool installed (pip install esptool)

echo "==============================================="
echo "BHR Battery Diagnostics - Firmware Flasher"
echo "==============================================="
echo ""

# Check if Python is installed
if ! command -v python3 &> /dev/null; then
    echo "ERROR: Python 3 is not installed"
    echo "Please install Python 3 using your package manager"
    exit 1
fi

# Check if esptool is installed
if ! python3 -m esptool version &> /dev/null; then
    echo "esptool not found. Installing..."
    pip3 install esptool
    if [ $? -ne 0 ]; then
        echo "ERROR: Failed to install esptool"
        exit 1
    fi
fi

echo ""
echo "Available serial ports:"
if command -v python3 -m serial.tools.list_ports &> /dev/null; then
    python3 -m serial.tools.list_ports
else
    ls /dev/ttyUSB* /dev/ttyACM* 2>/dev/null || echo "No devices found"
fi
echo ""

# Prompt for port
read -p "Enter serial port (e.g., /dev/ttyUSB0): " PORT

if [ -z "$PORT" ]; then
    echo "ERROR: No port specified"
    exit 1
fi

echo ""
echo "Flashing firmware to $PORT..."
echo "Please make sure your ESP32 is connected and no other program is using the port."
echo ""

# Flash the firmware
python3 -m esptool --chip esp32 --port "$PORT" --baud 460800 write_flash \
    0x1000 firmware/bootloader.bin \
    0x8000 firmware/partitions.bin \
    0xe000 firmware/boot_app0.bin \
    0x10000 firmware/firmware.bin

if [ $? -ne 0 ]; then
    echo ""
    echo "ERROR: Flashing failed!"
    echo "Make sure:"
    echo "  1. ESP32 is connected to $PORT"
    echo "  2. No other program (like serial monitor) is using the port"
    echo "  3. You selected the correct port"
    echo "  4. You have permissions to access the port (run with sudo if needed)"
    exit 1
fi

echo ""
echo "==============================================="
echo "SUCCESS! Firmware flashed successfully!"
echo "==============================================="
echo ""
echo "You can now disconnect and reconnect your device."
echo "Use a serial monitor at 115200 baud to communicate with the device."
echo ""
